import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import * as xlsx from 'xlsx'

// POST - Upload dan import data siswa dari Excel
export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData()
    const file = formData.get('file') as File

    if (!file) {
      return NextResponse.json(
        { error: 'File Excel harus diupload' },
        { status: 400 }
      )
    }

    // Baca file
    const arrayBuffer = await file.arrayBuffer()
    const workbook = xlsx.read(arrayBuffer, { type: 'array' })
    const sheetName = workbook.SheetNames[0]
    const sheet = workbook.Sheets[sheetName]

    // Convert ke JSON
    const jsonData = xlsx.utils.sheet_to_json(sheet, {
      defval: '',
      blankrows: false
    })

    if (jsonData.length === 0) {
      return NextResponse.json(
        { error: 'File Excel kosong' },
        { status: 400 }
      )
    }

    // Mapping kolom Excel ke field database
    const kolomMapping: { [key: string]: string } = {
      'Induk': 'induk',
      'NISN': 'nisn',
      'Tahun Masuk': 'tahunMasuk',
      'NIS Lokal': 'nisLokal',
      'Nama': 'nama',
      'L/P': 'jenisKelamin',
      'NIK': 'nik',
      'Nomor KK': 'nomorKk',
      'Agama': 'agama',
      'Tempat Lahir': 'tempatLahir',
      'Tanggal Lahir': 'tanggalLahir',
      'Anak Ke': 'anakKe',
      'Status Keluarga': 'statusKeluarga',
      'Telepon': 'telepon',
      'Alamat': 'alamat',
      'Kelas': 'kelas',
      'Status Siswa': 'statusSiswa',
      'Nama Ayah': 'namaAyah',
      'NIK Ayah': 'nikAyah',
      'TTL Ayah': 'ttlAyah',
      'Pendidikan Ayah': 'pendidikanAyah',
      'Pekerjaan Ayah': 'pekerjaanAyah',
      'Penghasilan Ayah': 'penghasilanAyah',
      'Nama Ibu': 'namaIbu',
      'NIK Ibu': 'nikIbu',
      'TTL Ibu': 'ttlIbu',
      'Pendidikan Ibu': 'pendidikanIbu',
      'Pekerjaan Ibu': 'pekerjaanIbu',
      'Penghasilan Ibu': 'penghasilanIbu',
      'Alamat Ortu': 'alamatOrtu'
    }

    // Process dan validasi data
    let successCount = 0
    let errorCount = 0
    const errors: string[] = []

    for (const row of jsonData as any[]) {
      try {
        // Map kolom
        const siswaData: any = {}
        for (const [excelKey, dbKey] of Object.entries(kolomMapping)) {
          if (row[excelKey] !== undefined && row[excelKey] !== '') {
            siswaData[dbKey] = row[excelKey]
          }
        }

        // Validasi field wajib
        if (!siswaData.induk || !siswaData.nama || !siswaData.kelas) {
          errors.push(`Baris yang error: ${JSON.stringify(row)}`)
          errorCount++
          continue
        }

        // Cek duplikasi
        const existing = await db.siswa.findUnique({
          where: { induk: siswaData.induk }
        })

        if (existing) {
          errors.push(`NIS induk ${siswaData.induk} sudah ada`)
          errorCount++
          continue
        }

        // Insert ke database
        await db.siswa.create({
          data: siswaData
        })

        successCount++
      } catch (error) {
        console.error('Error processing row:', error)
        errorCount++
      }
    }

    return NextResponse.json({
      success: true,
      message: `Berhasil mengimport ${successCount} data siswa${errorCount > 0 ? `, ${errorCount} gagal` : ''}`,
      stats: {
        success: successCount,
        error: errorCount,
        errors: errors.slice(0, 10) // Return max 10 errors
      }
    })
  } catch (error) {
    console.error('Upload siswa error:', error)
    return NextResponse.json(
      { error: 'Terjadi kesalahan saat mengupload file' },
      { status: 500 }
    )
  }
}
