import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'

// GET - Ambil semua data kelas
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const search = searchParams.get('search')

    const kelasList = await db.kelas.findMany({
      where: search
        ? {
            OR: [
              { namaKelas: { contains: search } },
              { kodeKelas: { contains: search } }
            ]
          }
        : undefined,
      orderBy: { kodeKelas: 'asc' }
    })

    return NextResponse.json({ success: true, data: kelasList })
  } catch (error) {
    console.error('Get kelas error:', error)
    return NextResponse.json(
      { error: 'Terjadi kesalahan saat mengambil data kelas' },
      { status: 500 }
    )
  }
}

// POST - Tambah kelas baru
export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { kodeKelas, namaKelas, waliKelas, keterangan } = body

    if (!kodeKelas || !namaKelas) {
      return NextResponse.json(
        { error: 'Kode kelas dan nama kelas harus diisi' },
        { status: 400 }
      )
    }

    const kelas = await db.kelas.create({
      data: {
        kodeKelas,
        namaKelas,
        waliKelas,
        keterangan
      }
    })

    return NextResponse.json({ success: true, data: kelas }, { status: 201 })
  } catch (error) {
    console.error('Create kelas error:', error)
    return NextResponse.json(
      { error: 'Terjadi kesalahan saat menambah kelas' },
      { status: 500 }
    )
  }
}
